import { NextRequest, NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';
import { categorySchema } from '@/lib/validations';
import { decrementUsage } from '@/lib/usage';

export async function PATCH(
  req: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session?.user) {
      return NextResponse.json({ error: 'Neautentificat' }, { status: 401 });
    }

    const restaurant = await prisma.restaurant.findFirst({
      where: { ownerId: session.user.id },
    });

    if (!restaurant) {
      return NextResponse.json({ error: 'Restaurant negăsit' }, { status: 404 });
    }

    const category = await prisma.category.findUnique({
      where: { id: params.id },
    });

    if (!category || category.restaurantId !== restaurant.id) {
      return NextResponse.json({ error: 'Categorie negăsită' }, { status: 404 });
    }

    const body = await req.json();
    const data = categorySchema.partial().parse(body);

    const updated = await prisma.category.update({
      where: { id: params.id },
      data,
    });

    await prisma.auditLog.create({
      data: {
        userId: session.user.id,
        action: 'UPDATE',
        entity: 'category',
        entityId: updated.id,
        metadata: { name: updated.nameRo },
      },
    });

    return NextResponse.json(updated);
  } catch (error) {
    console.error('Update category error:', error);
    return NextResponse.json({ error: 'Eroare server' }, { status: 500 });
  }
}

export async function DELETE(
  req: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session?.user) {
      return NextResponse.json({ error: 'Neautentificat' }, { status: 401 });
    }

    const restaurant = await prisma.restaurant.findFirst({
      where: { ownerId: session.user.id },
      include: { subscription: true },
    });

    if (!restaurant) {
      return NextResponse.json({ error: 'Restaurant negăsit' }, { status: 404 });
    }

    const category = await prisma.category.findUnique({
      where: { id: params.id },
    });

    if (!category || category.restaurantId !== restaurant.id) {
      return NextResponse.json({ error: 'Categorie negăsită' }, { status: 404 });
    }

    await prisma.category.delete({
      where: { id: params.id },
    });

    if (restaurant.subscription) {
      await decrementUsage(restaurant.subscription.id, 'categories', 1);
    }

    await prisma.auditLog.create({
      data: {
        userId: session.user.id,
        action: 'DELETE',
        entity: 'category',
        entityId: params.id,
        metadata: { name: category.nameRo },
      },
    });

    return NextResponse.json({ message: 'Categorie ștearsă' });
  } catch (error) {
    console.error('Delete category error:', error);
    return NextResponse.json({ error: 'Eroare server' }, { status: 500 });
  }
}